import pandas as pd
from cmdstanpy import CmdStanModel
import matplotlib.pyplot as plt
import numpy as np
import seaborn as sb
import scipy.stats as ss
import arviz as az

fout = open('Output.txt', 'w')

inflnm = input('Input data file (*.xlsx) =  ')
data_pf = pd.read_excel(inflnm)
print('data_pf =\n', data_pf)
data = data_pf.values
print('data =\n', data)

Data_Noise = data[:,1]
Data_Signal = data[:,2]
K = len(Data_Noise)
print('Data_Noise =\n', Data_Noise)
print('Data_Signal =\n', Data_Signal)

fout.write('\nData for Noise:\n')
fout.write('    {}\n'.format(Data_Noise))
fout.write('\nData for Signal:\n')
fout.write('    {}'.format(Data_Signal))


Data = {'K': K, 'Freq_Noise': Data_Noise, 'Freq_Signal': Data_Signal}

model = CmdStanModel(stan_file='YN_rating.stan')
fit = model.sample(data=Data)  

print(fit.diagnose())
print(fit.summary())

infdata = az.from_cmdstanpy(fit)  # Transforming to Arviz InfereceData
az.plot_trace(infdata, var_names=['mu', 'sigma'])  
plt.tight_layout()
plt.savefig('Fig_trace.png')
plt.show()

fit = fit.draws_pd()              #  Transforming to Pandas DataFrame
print(fit.keys())


mu = fit['mu']
sigma = fit['sigma']
mu_mean = mu.mean()
mu_q025, mu_med, mu_q975 = np.percentile(mu, [2.5, 50, 97.5]) # mu_s[int(len(mu_s) / 2)]
sgm_mean = sigma.mean()
sgm_q025, sgm_med, sgm_q975 = np.percentile(sigma, [2.5, 50, 97.5])  # sgm_s[int(len(sgm_s) / 2)]

fout.write('\nmu:\n')
fout.write('    med.     = {0:.3f}\n'.format(mu_med))
fout.write('    mean     = {0:.3f}\n'.format(mu_mean))
fout.write('    95% CI   = [{0:.3f}, {1:.3f}]\n'.format(mu_q025, mu_q975))

fout.write('\nsigma:\n')
fout.write('    med.     = {0:.3f}\n'.format(sgm_med))
fout.write('    mean     = {0:.3f}\n'.format(sgm_mean))
fout.write('    95% CI   = [{0:.3f}, {1:.3f}]\n'.format(sgm_q025, sgm_q975))

sb.kdeplot(mu)
plt.title((r'$\mu$') + (r'($\Delta$') + 'm) = ' +
           '{0:<.3f}(med)'.format(mu_med),
          fontsize = 18)
plt.yticks([])
plt.ylabel('Density', fontsize = 14)
plt.xlabel(r'$\mu$', fontsize = 16)
plt.tight_layout()
plt.savefig('Figure_mu.png')
plt.show()

sb.kdeplot(sigma)
plt.title(r"$\sigma$ = {0:<.3f}(med) ".format(sgm_med), fontsize = 18)
plt.yticks([])
plt.ylabel('Density', fontsize = 14)
plt.xlabel(r'$\sigma$', fontsize = 18)
plt.tight_layout()
plt.savefig('Figure_sgm.png')
plt.show()

c_smpls = []
for k in range(K-1):
    c_smpls.append(fit[f'c[{k+1}]'])
c_smpls = np.array(c_smpls).T

z_values = np.arange(-4.0, 4.001, 0.1)
x_values = 1.0 - ss.norm(0.0, 1.0).cdf(z_values)
y_values = 1.0 - ss.norm(mu_med, sgm_med).cdf(z_values)

plt.figure(figsize = (5,5))
plt.plot(x_values, y_values, 'b-')
plt.title(fr"ROC-curve($\Delta$m={mu_med:.3f}, $\sigma_s$={sgm_med:.3f})",
          fontsize = 18)
plt.xlabel("P(False Alarm)", fontsize = 14)
plt.ylabel("P(Hit)", fontsize = 14)

cum_freq_n = []
cum_freq_s = []
for k in range(K):
    cum_freq_n.append(Data_Noise[K - 1 - k])
    cum_freq_s.append(Data_Signal[K - 1 - k])

for k in range(1, K):
    cum_freq_n[k] += cum_freq_n[k-1]
    cum_freq_s[k] += cum_freq_s[k-1]

x_fa = []
y_ht = []
for k in range(K-1):
    x_fa.append(cum_freq_n[k] / cum_freq_n[K-1])
    y_ht.append(cum_freq_s[k] / cum_freq_s[K-1])
plt.plot(x_fa, y_ht, 'ro', label = 'Data')

print('\nI am calculating...')

c_means = np.zeros(K-1)
c_meds = np.zeros(K-1)
c_q025s = np.zeros(K-1)
c_q975s = np.zeros(K-1)
for k in range(K-1):
    c_means[k] = c_smpls.T[k].mean()
    c_q025s[k], c_meds[k], c_q975s[k] = \
                   np.percentile(c_smpls.T[k], [2.5, 50, 97.5])

fout.write('\n')
for k in range(K-1):
    fout.write('\nC{}:\n'.format(k+1))
    fout.write('    median   = {0:.3f}\n'.format(c_meds[k]))
    fout.write('    mean     = {0:.3f}\n'.format(c_means[k]))
    fout.write('    95% CI   = [{0:.3f}, {1:.3f}]\n'.format(
                    c_q025s[k], c_q975s[k]))
    
x_fa = []
y_ht = []

for k in range(K-1):
    x_fa.append(1.0 - ss.norm().cdf(c_meds[k]))  
    y_ht.append(1.0 - ss.norm(mu_med, sgm_med).cdf(c_meds[k]))  
plt.plot(x_fa, y_ht, 'gs', label = 'Prediction')

plt.legend(fontsize = 12)
plt.tight_layout()
plt.savefig('Figure_ROC.png')
plt.show()

for k in range(K-1):
    sb.kdeplot(c_smpls.T[k], label = '$C_{0}$: med = {1:.3f}'.
               format(k+1, c_meds[k]))
title_str = "Criterions" 
plt.title(title_str, fontsize = 18)
plt.yticks([])
plt.ylabel('Density', fontsize = 14)
plt.xlabel('C', fontsize = 14)
plt.legend(fontsize = 12)
plt.tight_layout()
plt.savefig('Figure_Cs.png')
plt.show()

fout.close()
print('Output.txt was saved.')
