import csv
import numpy as np
from cmdstanpy import CmdStanModel
import matplotlib.pyplot as plt
import seaborn as sb
import scipy.stats as ss
import arviz as az

fout = open('Results.txt', 'w')

fin_nm = input('Input file (*.csv) = ')
with open(fin_nm, 'r') as f:
    Data_in = [v for v in csv.reader(f)]

fout.write('Input Data File = {}\n'.format(fin_nm))

NSt = len(Data_in) - 1
K = len(Data_in[0]) - 1
Sts = np.empty(NSt)
Rs = np.empty((NSt, K), dtype = 'int')
for j in range(NSt):
    Sts[j] = float(Data_in[j+1][0])
    for k in range(K):
        Rs[j][k] = int(Data_in[j+1][k+1])

fout.write('\nData:\n')
for j in range(NSt):
    print(f'{Sts[j]:>10g}', end = '')
    fout.write(f'{Sts[j]:>10g}')
    for k in range(K):
        print(f'{Rs[j][k]:>5d}', end = '')
        fout.write(f'{Rs[j][k]:>5d}')
    print()
    fout.write('\n')

Data = {'NSt':NSt, 'K':K, 'Sts':Sts, 'Rs':Rs}

model = CmdStanModel(stan_file='RatingFechner.stan') 
fit = model.sample(data=Data)  

print(fit.diagnose())
print(fit.summary())

infdata = az.from_cmdstanpy(fit)  # Transforming to Arviz InfereceData
az.plot_trace(infdata, var_names=['Psis', 'C'])  
plt.tight_layout()
plt.savefig('Fig_trace.png')
plt.show()

fit = fit.draws_pd()              #  Transforming to Pandas DataFrame
print(fit.keys())

Psis = []
for s in range(NSt):
    Psis.append(fit[f'Psis[{s+1}]'])
Psis = np.array(Psis).T

sgm = fit['sgm']

Cs = []
for k in range(K-1):
    Cs.append(fit[f'C[{k+1}]'])
Cs = np.array(Cs).T

plt.figure(figsize=(14,4))
plt.subplot(131)

for i in range(NSt):
    sb.kdeplot(Psis.T[i], label = r'$\psi_{}$'.format(i+1))
plt.title(r'Posterior Ditributions of $\psi_s$', fontsize = 20)
plt.yticks([])
plt.legend()
plt.tight_layout()

plt.subplot(132)

sb.kdeplot(sgm)    
plt.title(r'Posterior Distribution of $\sigma$', fontsize = 20)
plt.legend()

plt.subplot(133)

plt.plot([0, 0], [0, 5], label = r'$C_0$')
for i in range(1, K-2):
    sb.kdeplot(Cs.T[i], label = r'$C_{}$'.format(i+1))
plt.plot([1,1], [0, 5], label = r'$C_{}$'.format(K-1))
plt.title('Posterior Distributions of Cs', fontsize = 20)
plt.yticks([])
plt.legend()
plt.savefig('FigCsPost.png')
plt.show()

plt.figure(figsize=(9,4))

plt.subplot(121)

v_psis = np.empty(NSt)
for i in range(NSt):
    v_psis[i] = np.median(Psis.T[i])  
v_sgm = np.median(sgm) 

fout.write('\n{0:>15s}{1:>15s}\n'.format('Stimulus', 'Med. Psi'))
for vst, vpsi in zip(Sts, v_psis):
    fout.write('{0:>15.2f}{1:>15.2f}\n'.format(vst, vpsi))
fout.write('\nSigma: med = {0:>15.2f}\n'.format(v_sgm))
    
v_Cs = np.empty(K-1)
v_Cs[0] = 0.0
v_Cs[K-2] = 1.0
for k in range(1, K-2):
    v_Cs[k] = np.median(Cs.T[k]) 
for k in range(K-1):
    plt.plot([Sts[0], Sts[-1]], [v_Cs[k], v_Cs[k]], color = 'y')
plt.plot([], 'y-', label = 'C')
plt.plot(Sts, v_psis, color = 'b', lw = 3, label = r'$\psi$')
plt.plot(Sts, v_psis - v_sgm, color = 'g', linestyle = '--', lw = 1,
         label = r'$\psi - \sigma$')
plt.plot(Sts, v_psis + v_sgm, color = 'g', linestyle = '--', lw = 1,
         label = r'$\psi+ \sigma$')
    
plt.xlabel('Stimulus', fontsize = 16)
plt.ylabel(r'$\psi$', fontsize = 16)
plt.legend(loc = 'upper left')
plt.title(r'Relation of Stimulus and $\psi$', fontsize = 18)
plt.tight_layout()

plt.subplot(122)

Probs = np.empty((NSt, K))
for s in range(NSt):
    Probs[s][0] = ss.norm(loc = v_psis[s], scale = v_sgm).cdf(v_Cs[0])
    Probs[s][K-1] = 1.0 - ss.norm(loc = v_psis[s], scale = v_sgm).cdf(v_Cs[K-2])
    for k in range(1, K-1):
        Probs[s][k] = ss.norm(loc = v_psis[s], scale = v_sgm).cdf(v_Cs[k]) -\
                        ss.norm(loc = v_psis[s], scale = v_sgm).cdf(v_Cs[k-1])
            
print('Predicted Prob(Rating|Stimulus):')
for s in range(NSt):
    for k in range(K):
        print(f'  {Probs[s][k]:7.3f}', end = '')
    print()
    
Pred_Rs = np.empty((NSt, K))
N_Rs = np.sum(Rs, axis = 1)
for i in range(NSt):
    Pred_Rs[i] = N_Rs[i] * Probs[i]
    
print('Predicted Frequencies of Rating for Stimulus:')
for s in range(NSt):
    for k in range(K):
        print(f'  {Pred_Rs[s][k]:7.1f}', end = '')
    print()

cum_data_Rs = np.cumsum(Rs, axis = 1)
cum_pred_Rs = np.cumsum(Pred_Rs, axis = 1)

print(cum_data_Rs)

for s in range(NSt):
    if s == 0:
        plt.plot(range(K), cum_data_Rs[s]/cum_data_Rs[s][K-1], 'g:', lw = 3,
                 label = 'Data')
    else:
        plt.plot(range(K), cum_data_Rs[s]/cum_data_Rs[s][K-1], 'g:', lw = 3 )
    if s == 0:
        plt.plot(range(K), cum_pred_Rs[s]/cum_pred_Rs[s][K-1], 'b-',
                 label = 'Model')
    else:
        plt.plot(range(K), cum_pred_Rs[s]/cum_pred_Rs[s][K-1], 'b-')

plt.xticks(range(K), range(1, K+1))
plt.xlabel('Rating', fontsize = 14)
plt.ylabel('Cum. Prop.', fontsize = 14)
plt.legend()
plt.title('Data and Model Prediction', fontsize = 18)
plt.tight_layout()
plt.savefig('FigDataModel.png')
plt.show()

fout.close()
print('Results.txt was saved.')
