from cmdstanpy import CmdStanModel
import pandas as pd
import matplotlib.pyplot as plt
import numpy as np
import seaborn as sb
import scipy.stats as ss
import arviz as az


outfnm = input('Output file (*.txt) = ')
fout = open(outfnm, 'w')  

inflnm = input('Input file (*.xlsx) = ')
df_data = pd.read_excel(inflnm)
Data_Noise = df_data.values.T[1]
Data_Signal = df_data.values.T[2]
K = len(Data_Noise)

print('Data_Noise =', Data_Noise)
print('Data_Signal =', Data_Signal)
print('K =', K)

fout.write('\nData for Noise:\n')
fout.write('    {}\n'.format(Data_Noise))
fout.write('\nData for Signal:\n')
fout.write('    {}'.format(Data_Signal))

Data = {'K': K, 'Freq_Noise': Data_Noise, 'Freq_Signal': Data_Signal}

model = CmdStanModel(stan_file="YN_rating.stan")    #   コンパイル
fit = model.sample(data=Data)                       #   サンプリング

print(fit.summary())
fout.write('\n{}'.format(fit.summary()))

inf_data = az.from_cmdstanpy(fit)           #   Arviz(az)用のデータに変換
az.plot_trace(inf_data, var_names=['mu', 'sigma', 'c'], figsize=(10,7))     #   トレースプロット
plt.tight_layout()
plt.show()

fit = fit.draws_pd()                        #   Pandas DataFrameへ変換

mu = fit['mu']
sigma = fit['sigma']
mu_mean = mu.mean()
mu_q025, mu_med, mu_q975 = np.percentile(mu, [2.5, 50, 97.5]) 
sgm_mean = sigma.mean()
sgm_q025, sgm_med, sgm_q975 = np.percentile(sigma, [2.5, 50, 97.5])  

fout.write('\nmu:\n')
fout.write('    med.     = {0:.3f}\n'.format(mu_med))
fout.write('    mean     = {0:.3f}\n'.format(mu_mean))
fout.write('    95% CI   = [{0:.3f}, {1:.3f}]\n'.format(mu_q025, mu_q975))

fout.write('\nsigma:\n')
fout.write('    med.     = {0:.3f}\n'.format(sgm_med))
fout.write('    mean     = {0:.3f}\n'.format(sgm_mean))
fout.write('    95% CI   = [{0:.3f}, {1:.3f}]\n'.format(sgm_q025, sgm_q975))

sb.kdeplot(mu)
plt.title((r'$\mu$') + (r'($\Delta$') + 'm) = ' +
           '{0:<.3f}(med)'.format(mu_med),
          fontsize = 18)
plt.yticks([])
plt.ylabel('Density', fontsize = 14)
plt.xlabel(r'$\mu$', fontsize = 16)
plt.tight_layout()
plt.show()

sb.kdeplot(sigma)
plt.title(r"$\sigma_s$ = {0:<.3f}(med) ".format(sgm_med), fontsize = 18)
plt.yticks([])
plt.ylabel('Density', fontsize = 14)
plt.xlabel(r'$\sigma_s$', fontsize = 18)
plt.tight_layout()
plt.show()

c = []
for k in range(K-1):
    c.append(fit[f'c[{k+1}]'])
c = np.array(c).T

c_comp = []
for i in range(3):
    c_comp.append([])

for v in c:
    for j in range(3):
        c_comp[j].append(v[j])

z_values = np.arange(-4.0, 4.001, 0.1)
x_values = 1.0 - ss.norm(0.0, 1.0).cdf(z_values)
y_values = 1.0 - ss.norm(mu_med, sgm_med).cdf(z_values)

plt.figure(figsize = (7, 7))
plt.plot(x_values, y_values, 'b-')
plt.title("ROC-curve", fontsize = 18)
plt.xlabel("P(False Alarm)", fontsize = 14)
plt.ylabel("P(Hit)", fontsize = 14)

cum_freq_n = []
cum_freq_s = []
for k in range(K):
    cum_freq_n.append(Data_Noise[K - 1 - k])
    cum_freq_s.append(Data_Signal[K - 1 - k])

for k in range(1, K):
    cum_freq_n[k] += cum_freq_n[k-1]
    cum_freq_s[k] += cum_freq_s[k-1]

x_fa = []
y_ht = []
for k in range(K-1):
    x_fa.append(cum_freq_n[k] / cum_freq_n[K-1])
    y_ht.append(cum_freq_s[k] / cum_freq_s[K-1])
plt.plot(x_fa, y_ht, 'ro', label = 'Data')

c_means = np.zeros(K-1)
c_meds = np.zeros(K-1)
c_q025s = np.zeros(K-1)
c_q975s = np.zeros(K-1)
for k in range(K-1):
    c_means[k] = c.T[k].mean()
    c_q025s[k], c_meds[k], c_q975s[k] = np.percentile(c.T[k], [2.5, 50, 97.5])

fout.write('\n')
for k in range(K-1):
    fout.write('\nC{}:\n'.format(k+1))
    fout.write('    median   = {0:.3f}\n'.format(c_meds[k]))
    fout.write('    mean     = {0:.3f}\n'.format(c_means[k]))
    fout.write('    95% CI   = [{0:.3f}, {1:.3f}]\n'.format(
                    c_q025s[k], c_q975s[k]))
    
x_fa = []
y_ht = []

for k in range(K-1):
    x_fa.append(1.0 - ss.norm().cdf(c_meds[k]))  
    y_ht.append(1.0 - ss.norm(mu_med, sgm_med).cdf(c_meds[k]))  
plt.plot(x_fa, y_ht, 'gs', label = 'Prediction')

plt.plot([], lw=0, label=fr"$\mu_s$={mu_med:.3f}" + "\n" +
                     fr"$\sigma_s$={sgm_med:.3f}")
plt.legend(fontsize = 12)
plt.tight_layout()
plt.show()

for k in range(K-1):
    sb.kdeplot(c.T[k], label = '$C_{0}$: med = {1:.3f}'.format(k+1, c_meds[k]))
title_str = "Criterions" 
plt.title(title_str, fontsize = 18)
plt.yticks([])
plt.ylabel('Density', fontsize = 14)
plt.xlabel('C', fontsize = 14)
plt.legend(fontsize = 12)
plt.tight_layout()
plt.show()

plt.figure(figsize=(9,5))
x_min = -3
x_max = 3 + mu_med
xcoord = np.arange(x_min, x_max, 0.01)
y_noise = ss.norm.pdf(xcoord, 0.0, 1.0)
plt.plot(xcoord, y_noise, lw=3, c='orange', label='noise')
y_signal = ss.norm.pdf(xcoord, mu_med, sgm_med)
plt.plot(xcoord, y_signal, lw=3, c='g', label='signal')
for k in range(K-1):
    plt.plot([c_meds[k], c_meds[k]], [0, 0.4], c='r', lw=2, ls='--'),

plt.plot([x_min, x_max], [0, 0], c='k', lw=2)
plt.plot([], ls='--', c='r', lw=2, label='Criterion')
plt.legend(fontsize=14)
plt.tight_layout()
plt.show()

fout.close()
print(f'\n{outfnm} was saved.\n')
